(**********************************************************************)
(*                                                                    *)
(* Function:    HALLD - Hall Effect Sensor Decode                     *)
(*                                                                    *)
(* Creation Date: 15/Oct/92                 From: NEW                 *)
(*                                                                    *)
(* Author:  Jeff Wright                                               *)
(*                                                                    *)
(* Description:                                                       *)
(* ------------                                                       *)
(* The HALLD function is designed to decode the sensor signals from a *)
(* 3 or 4 phase brushless motor. The two or three senor signals are   *)
(* decoded with a direction parameter supplied by the CPU into a state*)
(* No. This state number is then written to a programmable location   *)
(* and a link issued to the destination channel.                      *)
(* HALLD is specifically designed to work with the commutation [COMM] *)
(* TPU function.                                                      *)
(* The mapping of decoded state number to output pin configuration is *)
(* freely programmable in the state table of the COMM function.       *)
(*                                                                    *)
(*                                                                    *)
(* Updates:   By:   Modification:                                     *)
(* --------   ---   -------------                                     *)
(* 16/Jun/93  JW    Changed revNo from alpha to 1.0 - first release.  *)
(* 12/Aug/93  JL    Converted to new TPUMASM syntax.                  *)
(*                                                                    *)
(*--------------------------------------------------------------------*)
(* Standard Exits Used:-   End_Of_Phase: N         End_Of_Link: Y     *)
(*                                                                    *)
(* External Files Included: NONE.                                     *)
(*                                                                    *)
(* CODE SIZE excluding standard exits = 30 LONG WORDS                 *)
(*--------------------------------------------------------------------*)
(*                                                                    *)
(**********              This Revision:  1.1                  *********)
(*                                                                    *)
(**********   LAST MODIFIED: 12/Aug/93        BY: Jeff Loeliger *******)
(*                                                                    *)
(**********************************************************************)
(***************************************************************************)
(*Motorola reserves the right to make changes without further notice to any*)
(*product herein. Motorola makes no warranty, representation or guarantee  *)
(*regarding the suitability of its products for any particular purpose, nor*)
(*does Motorola assume any liability arising out of the application or use *)
(*of any product or circuit, and specifically disclaims any and all        *)
(*liability, including without limitation consequential or incidental      *)
(*damages. "Typical" parameters can and do vary in different applications. *)
(*All operating parameters, including "Typical",must be validated for each *)
(*customer application by customer's technical experts. Motorola does not  *)
(*convey any license under its patent rights nor the rights of others.     *)
(*Motorola products are not designed, intended, or authorized for use as   *)
(*components in systems intended for surgical implant into the body, or    *)
(*other applications intended to support or sustain life, or for any other *)
(*application in which the failure of the Motorola product could create a  *)
(*situation where injury or death may occur. Should Buyer purchase or use  *)
(*Motorola products for any such unintended or unauthorized application,   *)
(*Buyer, shall indemnify and hold Motorola and its officers, employees,    *)
(*subsidiaries, affiliates, and distributors harmless against all claims,  *)
(*costs, damages, and expenses, and reasonable attorney fees arising out   *)
(*of, directly or indirectly, any claim of personal injury or death        *)
(*associated with such unintended or unauthorized use, even if such claim  *)
(*alleges that Motorola was negligent regarding the design or manufacture  *)
(*of the part.                                                             *)
(*Motorola and the Motorola logo are registered trademarks of Motorola Inc.*) 
(*Motorola is an Equal Opportunity/Affirmative Action Employer.            *)
(*Copyright Motorola Inc. 1993                                             *)
(***************************************************************************)

(*()()()()()()()()()()() DATA STRUCTURE ()()()()()()()()()()()()()()()*)
(*                                                                    *)
(* name:               Written By:            Location  Bits(x..y):   *)
(* -----               -----------            ---------------------   *)
(* COMM_CHAN             CPU                 Parameter4   0..15       *)
(*                     Lower byte contains PRAM address of state No.  *)
(*                     parameter in the COMM channel. Also used to    *)
(*                     link to the COMM channel.                      *)
(*                                                                    *)
(* PINSTATE              TPU                 Parameter5   0..15       *)
(*                     Used to record current pin level [$0000/$8000] *)
(*                                                                    *)
(* DIRECTION             CPU                 Parameter3   0..15       *)
(*                     Controls the lsb of the decoded state number - *)
(*                     $0000 => msb=0, $0001 => msb=1. Effectively for*)
(*                     each decoded input state, DIRECTION selects one*)
(*                     of two possible output states from COMM.       *)
(*                                                                    *)
(*                                                                    *)
(* hsr1   hsr0         Action                                         *)
(* ----   ----         ------                                         *)
(*  0       X          No action                                      *)
(*  1       0          Initialise - 2 sensor operation                *)
(*  1       1          Initialise - 3 sensor operation                *)
(*                                                                    *)
(*                                                                    *)
(* HSQ1   HSQ0         Action                                         *)
(* ----   ----         ------                                         *)
(*  0       0          Sensor 'A' channel.                            *)
(*  0       1          Sensor 'B' channel.                            *)
(*  1       1          Sensor 'C' channel- 3 sensor operation only.   *)
(*                                                                    *)
(* Links Accepted: no              Links Generated: YES               *)
(*                                                                    *)
(* Interrupts Generated After:     none                               *)
(*                                                                    *)
(*()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()*)


(*+++++++++++++++++++++  PARAMETER MACROS  +++++++++++++++++++++++++++*)

%macro DIRECTION_HALLD       'prm3'.
%macro COMM_CHAN_HALLD       'prm4'.
%macro PINSTATE_HALLD        'prm5'.


(*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)


(*====================================================================*)
(*|||||||||||||||||||||  MICROCODE STARTS BELOW  |||||||||||||||||||||*)
(*VVVVVVVVVVVVVVVVVVVVV--------------------------VVVVVVVVVVVVVVVVVVVVV*)


(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  INIT_HALLD                                            *)
(*                                                                    *)
(* STATE(S) ENTERED: S1                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : NONE                                           *)
(*                                                                    *)
(* ENTER WHEN : HSR = %10                                             *)
(*                                                                    *)
(* ACTION: Initialise, 2 channel mode. Decode & link.                 *)
(*                                                                    *)
(**********************************************************************)
%entry  start_address *; disable_match;
name = INIT_HALLD;
cond hsr1 = 1, hsr0 = 0.

init_halld:
         If true then goto chk_pin_halld,flush;
         chan pac := any_trans, tbs := in_m1_c1, enable_mtsr;
         chan clear flag0.



(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  INIT_A_HALLD                                          *)
(*                                                                    *)
(* STATE(S) ENTERED: S2                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : NONE                                           *)
(*                                                                    *)
(* ENTER WHEN : HSR = %11                                             *)
(*                                                                    *)
(* ACTION: Initialise, 3 channel mode. Decode & link.                 *)
(*                                                                    *)
(**********************************************************************)
%entry  start_address *; disable_match;
name = INIT_A_HALLD;
cond hsr1 = 1, hsr0 = 1.

init_A_halld:
         chan pac := any_trans, tbs := in_m1_c1, enable_mtsr;
         chan set flag0.




(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  TRANS_HALLD                                           *)
(*                                                                    *)
(* STATE(S) ENTERED: S3                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : none                                           *)
(*                                                                    *)
(* ENTER WHEN : m/tsr = 1                                             *)
(*                                                                    *)
(* ACTION: Decode state number and store in remote location, then link*)
(*                                                                    *)
(**********************************************************************)
%entry start_address *; disable_match;
name = EDGE_HALLD;
cond hsr1 = 0, hsr0 = 0,m/tsr = 1,lsr = x.

chk_pin_halld:
         au chan_reg := chan_reg;                  (* latch latest pinstate *)
         neg_tdl, neg_mrl, neg_lsl.               (* - and any future edges *)

         au p := max;
         ram p -> @PINSTATE_HALLD.

         If PSL = 1 then goto chan_ident_halld,flush. (* store new pinstate *)

         au p := 0;
         ram p -> @PINSTATE_HALLD.


                       (* identify channel and move to chan A if neccessary *)
chan_ident_halld:
         If hsq1 = 0 then goto tst_S_B_halld,flush.

         au chan_reg := chan_reg + #$F0.             (* move down a channel *)

tst_S_B_halld:
         If hsq0 = 0 then goto ld_sr_halld,flush.

         au chan_reg := chan_reg + #$F0.

ld_sr_halld:
         au sr :=<< 1;                                  (* Now at channel A *)
         ram p <- @DIRECTION_HALLD.      (* DIRECTION must = $0000 or $0001 *)

start_decode_halld:
         call chan_sub_halld,flush;                    (* Go test channel A *)
         ram diob <- @PINSTATE_HALLD.

         call chan_sub_halld,flush;                    (* Go test channel B *)
         ram diob <- @PINSTATE_HALLD.

         If flag0 = 0 then goto got_num_halld,flush.    (* 2 or 3 sensors ? *)

         call chan_sub_halld,flush;                    (* Go test channel C *)
         ram diob <- @PINSTATE_HALLD.

got_num_halld:
         au link := diob;          (* link to COMM channel & store state No *)
         ram p -> by_diob;
         end.                                          (* END function here *)



(*--------------------- Subroutine CHAN_SUB --------------------------------*)
(* enter with diob = chan pinstate*)


chan_sub_halld:
         au nil := diob,ccl;             (* test channel pinstate parameter *)
         ram diob <- @COMM_CHAN_HALLD.

         If N = 0 then goto ret_sub_halld,flush.

         au p := p + sr.             (* add current 'adder' if pin was high *)

ret_sub_halld:
         au sr :=<< sr.      (* multiply 'adder' by 2 for next channel test *)

         return.

         au chan_reg := chan_reg + #$10.            (* move to next channel *)



(**********************************************************************)
(*    UNUSED ENTRIES - execute appropriate termination                *)
(**********************************************************************)

%entry  start_address  End_of_link;
name = del_lnk_halld1;
cond hsr1 = 0,hsr0 = 0,m/tsr = 0,lsr = 1.

%entry  start_address  End_of_link;
name = del_lnk_halld2;
cond hsr1 = 0,hsr0 = 1.
