(**********************************************************************)
(*                                                                    *)
(* Function:    NITC  Input Capture/Transition Counter                *)
(*                                                                    *)
(* Creation Date: 09/Nov/92                 From: ITC                 *)
(*                                                                    *)
(* Author:  Jeff Wright                                               *)
(*                                                                    *)
(* Description:                                                       *)
(* ------------                                                       *)
(* Capture TCR1 or TCR2 or a PRAM location on a user defined input pin*)
(* transition. Store captured value in PRAM. Repeat this process for  *)
(* a user defined number of transitions then store the last captured  *)
(* value in a differnt parameter,interrupt the CPU and optionally link*)
(* to a block of channels. If continuous mode then start entire       *)
(* process over again, otherwise stop.                                *)
(* Note: If the 'PRAM capture' mode is enabled, the parameter ram     *)
(* value is not actually captured at the time of the pin transition,  *)
(* but during the service of the channel. In 'TCRx capture' mode, the *)
(* specified TCR will be automatically latched at the time of the pin *)
(* transition.                                                        *)
(*                                                                    *)
(* Updates:   By:   Modification:                                     *)
(* --------   ---   -------------                                     *)
(* 16/Jun/93   JW   Changed revNo from alpha to 1.0 - first release   *)
(* 12/Aug/93   JL   Converted to new TPUMASM syntax.                  *)
(*                                                                    *)
(*--------------------------------------------------------------------*)
(* Standard Exits Used:-   End_Of_Phase: N         End_Of_Link: Y     *)
(*                                                                    *)
(* External Files included: LINKCHAN.                                 *)
(*                                                                    *)
(* CODE SIZE excluding standard exits = 35 LONG WORDS [incl linkchan] *)
(*--------------------------------------------------------------------*)
(*                                                                    *)
(*                                                                    *)
(**********              This Revision:  1.1                  *********)
(*                                                                    *)
(**********   LAST MODIFIED: 12/Aug/93       BY: Jeff Loeliger ********)
(*                                                                    *)
(**********************************************************************)
(***************************************************************************)
(*Motorola reserves the right to make changes without further notice to any*)
(*product herein. Motorola makes no warranty, representation or guarantee  *)
(*regarding the suitability of its products for any particular purpose, nor*)
(*does Motorola assume any liability arising out of the application or use *)
(*of any product or circuit, and specifically disclaims any and all        *)
(*liability, including without limitation consequential or incidental      *)
(*damages. "Typical" parameters can and do vary in different applications. *)
(*All operating parameters, including "Typical",must be validated for each *)
(*customer application by customer's technical experts. Motorola does not  *)
(*convey any license under its patent rights nor the rights of others.     *)
(*Motorola products are not designed, intended, or authorized for use as   *)
(*components in systems intended for surgical implant into the body, or    *)
(*other applications intended to support or sustain life, or for any other *)
(*application in which the failure of the Motorola product could create a  *)
(*situation where injury or death may occur. Should Buyer purchase or use  *)
(*Motorola products for any such unintended or unauthorized application,   *)
(*Buyer, shall indemnify and hold Motorola and its officers, employees,    *)
(*subsidiaries, affiliates, and distributors harmless against all claims,  *)
(*costs, damages, and expenses, and reasonable attorney fees arising out   *)
(*of, directly or indirectly, any claim of personal injury or death        *)
(*associated with such unintended or unauthorized use, even if such claim  *)
(*alleges that Motorola was negligent regarding the design or manufacture  *)
(*of the part.                                                             *)
(*Motorola and the Motorola logo are registered trademarks of Motorola Inc.*) 
(*Motorola is an Equal Opportunity/Affirmative Action Employer.            *)
(*Copyright Motorola Inc. 1993                                             *)
(***************************************************************************)


(*()()()()()()()()()()() DATA STRUCTURE ()()()()()()()()()()()()()()()*)
(*                                                                    *)
(* name:               Written By:            Location  Bits:         *)
(* -----               -----------            ---------------         *)
(* CHAN_CONTROL_NITC      CPU                 Parameter0  0..8        *)
(*                     9 bit parameter determines which TCR1 will be  *)
(*                     captured and on what edge type. Must be input. *)
(*                                                                    *)
(* PARAM_ADDR_NITC        CPU                 Parameter1  0..7        *)
(*                     Contains the PRAM address of the parameter to  *)
(*                     "captured" if PRAM capture mode enabled.[HSR10]*)
(*                                                                    *)
(* LINK_CHAN_CNT_NITC     CPU                 Parameter1  8..11       *)
(*                     The number of channels to be linked to if links*)
(*                     are enabled. Must be between 0 and 8.          *)
(*                                                                    *)
(* START_LINK_CHAN_NITC   CPU                 Parameter1  12..15      *)
(*                     The channel number of the first channel in the *)
(*                     block of channles to be linked.                *)
(*                                                                    *)
(* MAX_CNT_NITC           CPU                 Parameter2  0..15       *)
(*                     The number of transitions to be counted before *)
(*                     linking and interrupting CPU.                  *)
(*                                                                    *)
(* TRANS_CNT_NITC        BOTH                 Parameter3  0..15       *)
(*                     Counter for number of transitions detected.    *)
(*                                                                    *)
(* FINAL_TIM_NITC         TPU                 Parameter4  0..15       *)
(*                     The captured TCR time or PRAM value correspond-*)
(*                     ing to the final pin transition.               *)
(*                                                                    *)
(* LAST_TIM_NITC          TPU                 Parameter5  0..15       *)
(*                     The captured TCR time or PRAM value correspond-*)
(*                     ing to the latest pin transition. - not updated*)
(*                     on final transition.                           *)
(*                                                                    *)
(* hsr1   hsr0         Action                                         *)
(* ----   ----         ------                                         *)
(*  0       0          No Action.                                     *)
(*  0       1          Initialise - TCR capture mode.                 *)
(*  1       0          Initialise - Pram capture mode.                *)
(*  1       1          No action.                                     *)
(*                                                                    *)
(* HSQ1   HSQ0         Action                                         *)
(* ----   ----         ------                                         *)
(*  0       0          Single shot mode,no links.                     *)
(*  0       1          Continuous mode, no links.                     *)
(*  1       0          Single shot mode, links.                       *)
(*  1       1          Continuous mode, links.                        *)
(*                                                                    *)
(* Links Accepted: NO             Links Generated: YES                *)
(*                                                                    *)
(* Interrupts Generated After:     MAX_CNT transitions detected.      *)
(*                                                                    *)
(*()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()*)


(*+++++++++++++++++++++  PARAMETER MACROS  +++++++++++++++++++++++++++*)

%macro CHAN_CONTROL_NITC     'prm0'.
%macro PARAM_ADDR_NITC       'prm1'.
%macro LINK_CHAN_CNT_NITC    'prm1'.
%macro START_LINK_CHAN_NITC  'prm1'.
%macro MAX_CNT_NITC          'prm2'.
%macro TRANS_CNT_NITC        'prm3'.
%macro FINAL_TIM_NITC        'prm4'.
%macro LAST_TIM_NITC         'prm5'.


(*====================================================================*)
(*|||||||||||||||||||||  MICROCODE STARTS BELOW  |||||||||||||||||||||*)
(*VVVVVVVVVVVVVVVVVVVVV--------------------------VVVVVVVVVVVVVVVVVVVVV*)


%include 'linkchan.UC'.

(* Include routine to link to a block of channels. *)


(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  COUNT_NITC                                            *)
(*                                                                    *)
(* STATE(S) ENTERED: S2                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : START_LINK_CHAN                                *)
(*                                                                    *)
(* ENTER WHEN : a pin transition has occured.                         *)
(*                                                                    *)
(* ACTION: Update transcount. Store param or captured TCR in PRAM.    *)
(*         if trans_cnt = max_cnt then interrupt and optionally link. *)
(*         - if single shot mode then stop further transition detects *)
(*         if trans_cnt = max_cnt.                                    *)
(*                                                                    *)
(**********************************************************************)
%entry  ram p <- @START_LINK_CHAN_NITC; start_address *;
name = count_NITC;
cond hsr1 = 0,hsr0 = 0,m/tsr = 1,lsr = 0.

Count_NITC :
					(* increment TRANS_COUNT coherently *)
       au dec := p_high;                            (* prapare dec for link *)
       ram diob <- @TRANS_CNT_NITC.

       au diob := diob  + 1;
       ram diob -> @TRANS_CNT_NITC.

				     (* read coherently TRANSCNT and MACXNT *)
       au diob := p_low;
       ram p <- @TRANS_CNT_NITC.

       au sr := p;
       ram p <- @MAX_CNT_NITC.               (* compare transcnt and maxcnt *)

       If flag0 = 1 then goto tst_cc_NITC.     (* store PRAM value or ert ? *)

       au nil := p - sr, ccl;
       ram diob <- by_diob.

       au diob := ert.

TST_CC_NITC:
       if LOW_SAME = TRUE then goto Count_ended_NITC, flush.

       goto end_NITC,flush;                   (* still counting transitions *)
       ram diob -> @LAST_TIM_NITC.


COUNT_ENDED_NITC:
       if hsq1 = 0 then goto check_single_mode_NITC.

       ram diob -> @FINAL_TIM_NITC.              (* update FINAL_TRANS_TIME *)


       call Link_chan, flush; dec_return;   (* link to one or more channels *)
       ram p <- @START_LINK_CHAN_NITC.


CHECK_SINGLE_MODE_NITC:
       if hsq0 = 1 then goto reset_cnt_NITC.

       chan cir.               (* interrupt after every MAX_CNT transitions *)

       if true then goto end_nitc,flush;
       chan PAC := no_detect.               (* Disable further edge capture *)


(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  INIT_NITC                                             *)
(*                                                                    *)
(* STATE(S) ENTERED: S0                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : CHAN_CONTROL                                   *)
(*                                                                    *)
(* ENTER WHEN : HSR= %01.                                             *)
(*                                                                    *)
(* ACTION: Initialise channel, reset trans_cnt to $0000. TCR capture. *)
(*                                                                    *)
(**********************************************************************)
%entry  ram p <- @CHAN_CONTROL_NITC; start_address *;
name = init_NITC;
cond hsr1 = 0, hsr0 = 1.

INIT_NITC:
	 If true then goto reset_cnt_NITC,flush;
	 chan config := p, enable_mtsr;      (* configure channel from PRAM *)
	 chan clear flag0.



(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  INITA_NITC                                            *)
(*                                                                    *)
(* STATE(S) ENTERED: S1                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : CHAN_CONTROL                                   *)
(*                                                                    *)
(* ENTER WHEN : HSR= %10.                                             *)
(*                                                                    *)
(* ACTION: Initialise channel, reset trans_cnt to $0000. Param capture*)
(*                                                                    *)
(**********************************************************************)
%entry  ram p <- @CHAN_CONTROL_NITC; start_address *;
name = inita_NITC;
cond hsr1 = 1, hsr0 = 0.

INITA_NITC :
	 chan config := p, enable_mtsr;      (* configure channel from PRAM *)
	 chan set flag0.

RESET_CNT_NITC:
	 au diob := 0;                          (* reset transition counter *)
	 ram diob -> @TRANS_CNT_NITC.


END_NITC:
	 chan neg_lsl, neg_mrl, neg_tdl;    (* Clear all pending conditions *)
	 end.


(**********************************************************************)
(*     UNUSED ENTRIES - execute appropriate termination.              *)
(**********************************************************************)
%entry  start_address End_of_link;
name = nitc_undef;
cond hsr1 = 0,hsr0 = 0,m/tsr = 0,lsr =1.

%entry  start_address End_of_link;
name = nitc_undef;
cond hsr1 = 1,hsr0 = 1.

%entry  start_address End_of_link;
name = nitc_undef;
cond hsr1 = 0,hsr0 = 0,m/tsr = 1,lsr =1.

