(**********************************************************************)
(*                                                                    *)
(* Function:    PTA -   PROGRAMMABLE TIME ACCUMULATOR FUNCTION        *)
(*                                                                    *)
(* Creation Date: 13/03/92                  From: NEW                 *)
(*                                                                    *)
(* Author:  Richard Soja                                              *)
(*                                                                    *)
(* Description:                                                       *)
(* ------------                                                       *)
(* Provides Pulse width measurement or Period measurement.            *)
(* Two different types of pulse width measurement are supported:      *)
(* Either the high pulse width or the low pulse width is measured.    *)
(* Two different types of period measurement are supported:           *)
(* The period measurement is updated either on the rising edge of the *)
(* waveform or on the falling edge.                                   *)
(*                                                                    *)
(* Notes for revisions 0.0, 0.1, 0.2                                  *)
(* ---------------------------------                                  *)
(* When switching modes using an HSR to initialise the PTA, it is     *)
(* possible that there is a remaining pending match from the previous *)
(* operating mode. The init state does not clear pending matches.     *)
(* This results in possible transient erroneous operation of the PTA  *)
(* function after initialisation from a previous mode.                *)
(* Note that Rev 0.3 fixes these problems!                            *)
(*                                                                    *)
(*    Mode switch                       Effect                        *)
(*    -----------                       ------                        *)
(*   Any to High or Low Pulse      Pending match is discarded in      *)
(*                                 initial (CFLAG0=1) states,         *)
(*                                 regardless of pin state.           *)
(*                                                                    *)
(*   Any to Rising Edge Period,                                       *)
(*   pin = 0.                      Causes continuous match service    *)
(*                                 until rising edge occurs.          *)
(*                                 Introduces 1.2uS overhead on TPU.  *)
(*   pin = 1.                      Causes period measurement to start *)
(*                                 at match time.                     *)
(*   Any to Falling Edge Period,                                      *)
(*   pin = 0.                      Causes period measurement to start *)
(*                                 at match time.                     *)
(*   pin = 1.                      Causes continuous match service    *)
(*                                 until falling edge occurs.         *)
(*                                 Introduces 1.2uS overhead on TPU.  *)
(*                                                                    *)
(*   Note: Match time is fixed at $8000 - equivalent to 7.86mS with   *)
(*         240nS resolution. Fault does not occur if period is less   *)
(*         than this.                                                 *)
(*         If period exceeds $8000 TCR counts, then 1st measurement   *)
(*         after re-initialisation should be discarded.               *)
(*                                                                    *)
(*                                                                    *)
(*--------------------------------------------------------------------*)
(* Standard Exits Used:-   End_Of_Phase: Y         End_Of_Link: Y     *)
(*                                                                    *)
(* Total size (including standard exits):                             *)
(* Code    =    55                                                    *)
(* Entries =     8                                                    *)
(* Total   =    63                                                    *)
(*                                                                    *)
(* This Revision:  1.1                                                *)
(*                                                                    *)
(**********   LAST MODIFIED: 12/Aug/93        BY: Jeff Loeliger *******)
(*                                                                    *)
(*  Revision 0.1 : 7/4/92 R.Soja                                      *)
(*                 Added <clear_all_flags_pta> label at               *)
(*                 <init_flag0_done_pta> in init_pta state.           *)
(*                 Changed start_address address for link states.     *)
(*  Revision 0.2 : 9/4/92 R.Soja                                      *)
(*                 Deleted 2nd Period_Measurement test in State 3.    *)
(*                 Added actual pin level tests to avoid bad          *)
(*                 measurements due to TPU latency.                   *)
(*  Revision 0.3 : 6/5/93 R.Soja                                      *)
(*                 Fixed problems described in notes above.           *)
(*                 Reduced coding for period measurement.             *)
(*  Rev 1.0        16/Jun/93 JW - changed revNo for mask release.     *)
(*  Rev 1.1        12/Aug/93 JL - Converted to new TPUMASM syntax.    *)
(*                                                                    *)
(**********************************************************************)
(***************************************************************************)
(*Motorola reserves the right to make changes without further notice to any*)
(*product herein. Motorola makes no warranty, representation or guarantee  *)
(*regarding the suitability of its products for any particular purpose, nor*)
(*does Motorola assume any liability arising out of the application or use *)
(*of any product or circuit, and specifically disclaims any and all        *)
(*liability, including without limitation consequential or incidental      *)
(*damages. "Typical" parameters can and do vary in different applications. *)
(*All operating parameters, including "Typical",must be validated for each *)
(*customer application by customer's technical experts. Motorola does not  *)
(*convey any license under its patent rights nor the rights of others.     *)
(*Motorola products are not designed, intended, or authorized for use as   *)
(*components in systems intended for surgical implant into the body, or    *)
(*other applications intended to support or sustain life, or for any other *)
(*application in which the failure of the Motorola product could create a  *)
(*situation where injury or death may occur. Should Buyer purchase or use  *)
(*Motorola products for any such unintended or unauthorized application,   *)
(*Buyer, shall indemnify and hold Motorola and its officers, employees,    *)
(*subsidiaries, affiliates, and distributors harmless against all claims,  *)
(*costs, damages, and expenses, and reasonable attorney fees arising out   *)
(*of, directly or indirectly, any claim of personal injury or death        *)
(*associated with such unintended or unauthorized use, even if such claim  *)
(*alleges that Motorola was negligent regarding the design or manufacture  *)
(*of the part.                                                             *)
(*Motorola and the Motorola logo are registered trademarks of Motorola Inc.*) 
(*Motorola is an Equal Opportunity/Affirmative Action Employer.            *)
(*Copyright Motorola Inc. 1993                                             *)
(***************************************************************************)


(*()()()()()()()()()()() DATA STRUCTURE ()()()()()()()()()()()()()()()*)
(*                                                                    *)
(* name:               Access:            Description:                *)
(* -----               -------            ------------                *)
(* CONFIG_PTA           CPU    Param0[0] selects TCR1 or TCR2 timebase*)
(*                                                                    *)
(* START_LINK/          TPU    Param0[8..15] Unused in this version.  *)
(* LINK_COUNT_PTA                                                     *)
(*                                                                    *)
(* MAX_COUNT_PTA       CPU/TPU Param1[8..15] Number of periods over   *)
(*                             which measurement is made.             *)
(*                                                                    *)
(* PERIOD_COUNT_PTA     TPU    Param1[0..7] Elapsed periods.          *)
(*                                                                    *)
(* LAST_TIME_PTA        TPU    Param2[0..15] Time of last event.      *)
(*                                                                    *)
(* ACCUM_PTA            TPU    Param3[0..15] Uncompleted measurement. *)
(*                                                                    *)
(* HW_PTA             CPU/TPU  Param4[0..15] High word value of       *)
(*                             completed measurement.                 *)
(*                                                                    *)
(* LW_PTA             CPU/TPU  Param5[0..15] Low word value of        *)
(*                             completed measurement.                 *)
(*                                                                    *)
(* HSQ1   HSQ0         Action                                         *)
(* ----   ----         ------                                         *)
(*  0       0          Measure High Pulse Width.                      *)
(*  0       1          Measure Low Pulse Width.                       *)
(*  1       0          Measure Period on Rising Edge.                 *)
(*  1       1          Measure Period on Falling Edge.                *)
(*                                                                    *)
(* Links Accepted: NO              Links Generated: NO                *)
(*                                                                    *)
(* Interrupts Generated After: Period_Count >= Max_Count              *)
(*                                                                    *)
(*                                                                    *)
(*()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()*)


(*+++++++++++++++++++++  PARAMETER MACROS  +++++++++++++++++++++++++++*)

%macro CONFIG_PTA            'prm0'.
%macro START_COUNT_LINK_PTA  'prm0'.
%macro MAX_COUNT_PTA         'prm1'.    (* MS byte *)
%macro PERIOD_MAX_PTA        'prm1'.
%macro LAST_TIME_PTA         'prm2'.
%macro ACCUM_PTA             'prm3'.
%macro HW_PTA                'prm4'.
%macro LW_PTA                'prm5'.

(*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)

(*+++++++++++++++++++++  CONSTANTS MACROS  +++++++++++++++++++++++++++*)

%macro Period_Measure           'hsq1 = 1'.
%macro Pulse_Measure            'hsq1 = 0'.
%macro Low_Pulse                'hsq0 = 1'.
%macro High_Pulse               'hsq0 = 0'.
%macro Falling_Period           'hsq0 = 1'.
%macro Rising_Period            'hsq0 = 0'.
%macro Match                    'MRL  = 1'.
%macro Not_Match                'MRL  = 0'.
%macro Not_Overflow             'C    = 0'.
%macro TCR1_Selected            'N    = 0'.
%macro Not_Limit                'N    = 0'.



(*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)

(*====================================================================*)
(*|||||||||||||||||||||  MICROCODE STARTS BELOW  |||||||||||||||||||||*)
(*VVVVVVVVVVVVVVVVVVVVV--------------------------VVVVVVVVVVVVVVVVVVVVV*)


(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  INIT_PTA                                              *)
(*                                                                    *)
(* STATE(S) ENTERED: S0                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : CONFIG_PTA                                     *)
(*                                                                    *)
(* ENTER WHEN : HSR = %11                                             *)
(*                                                                    *)
(* ACTION:                                                            *)
(*         Initialises channel timebase, pin action control, flag0.   *)
(*                                                                    *)
(**********************************************************************)
%entry  ram p <- @CONFIG_PTA; start_address *; disable_match;
name = init_pta;
cond hsr1 = 1, hsr0 = 1.

init_pta:
        chan config := p, enable_mtsr; chan clear flag0;
        if @Period_Measure then goto init_flag0_done_pta, flush.
        chan set flag0.

init_flag0_done_pta :

(**** For test purposes only ****)
(*        call disp_state_pta, flush.
        chan cir;*)
(********************************)

clear_all_flags_pta :
        chan neg_lsl, neg_mrl, neg_tdl;
        end.

(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  POS_TRANSITION                                        *)
(*                                                                    *)
(* STATE(S) ENTERED: S1                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : P <- LAST_TIME                                 *)
(*                                                                    *)
(* ENTER WHEN: m/tsr = 1, flag0 = 0                                   *)
(*                                                                    *)
(* ACTION :          On Transition                    On Match        *)
(* High Pulse     : Null.                       Update measurement.   *)
(* Low Pulse      : Update measurement.         Invalid.              *)
(* Rising Period  : Start measurement.          Null.                 *) (*V0.3 R.Soja, 6/5/93 *)
(* Falling Period : Start measurement (Note).   Null.                 *) (*V0.3 R.Soja, 6/5/93 *)
(*                                                                    *)
(* Note: Caused by short negative pulse.                              *) (*V0.3 R.Soja, 6/5/93 *)
(**********************************************************************)
%entry  ram p <- @LAST_TIME_PTA; start_address *; disable_match;
name = pos_trans0;
cond hsr1 = 0, hsr0 = 0, lsr = x, m/tsr = 1, pin = 1, flag0 = 0.

pos_trans0 :

        if @Period_Measure then goto next_match1_pta, flush. (*V0.3 R.Soja, 6/5/93 *)
        if @Match then goto accum_time_pta, flush. (* Can only be High Pulse *)
        au chan_reg := chan_reg;  (* It's O.K. to update PSL here since *)
        chan neg_tdl.             (* there are no more Match states.    *)
        nop.                      (*V0.3 R.Soja, 6/5/93 *)

(**************************************************************) (*V0.3 R.Soja, 6/5/93 *)
(* Extra code to check for short pulses during flag0=0 states *) (*V0.3 R.Soja, 6/5/93 *)
(* in High and Low Pulse measurement functions.               *) (*V0.3 R.Soja, 6/5/93 *)
(**************************************************************) (*V0.3 R.Soja, 6/5/93 *)
low_pulse10_pta :                                                (*V0.3 R.Soja, 6/5/93 *)
        if PSL = 0 then goto End_Of_Phase, flush.                 (*V0.3 R.Soja, 6/5/93 *)
        goto low_pulse0_pta, flush.                               (*V0.3 R.Soja, 6/5/93 *)

accum_time1_pta :
        if @Match then goto accum_time_pta, flush. (*V0.3 R.Soja, 6/5/93 *)
        chan neg_tdl.

accum_time_pta :
        au p := ert - p;
        ram diob <- @ACCUM_PTA.

        au diob := diob + p, ccl;
        ram diob -> @ACCUM_PTA.

        if @Not_Overflow then goto no_overflow_pta, flush.
        ram p <- @HW_PTA.
        au p := p + 1, ccl;
        ram p -> @HW_PTA.
        if @Not_Limit then goto no_overflow_pta, flush.
        chan cir.

no_overflow_pta :
        if @Match then goto next_match_pta, flush.

check_count_pta :

        ram p <- @PERIOD_MAX_PTA.
        au p_low := p_low + 1;
        ram p -> @PERIOD_MAX_PTA.
        au p_low := p_high - p, ccl.
        if LOW_SAME = 0 then goto next_match_pta, flush.
        au p_low := 0;
        ram p -> @PERIOD_MAX_PTA.

        ram diob -> @LW_PTA.

(**** Note: Remove for test purposes only ****)
        chan cir.
(*********************************************)

        au p := 0;
        ram p -> @ACCUM_PTA.

next_match_pta :
        au p := ert;
        ram p -> @LAST_TIME_PTA.

(**** For test purposes only ****)
(*        call disp_state_pta, flush.
        chan cir;*)
(********************************)

        au ert := ert + max;
        chan write_mer, neg_mrl;
        end.

(*        if @Falling_Period then goto end_state_pta.        *)  (*V0.3 R.Soja, 6/5/93 *)

next_match1_pta :
        if @Match then goto clear_match_pta, flush.               (*V0.3 R.Soja, 6/5/93 *)
        chan neg_tdl.
        chan set flag0;
        goto next_match_pta, flush.

(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  POS_TRANSITION                                        *)
(*                                                                    *)
(* STATE(S) ENTERED: S2                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : P <- LAST_TIME                                 *)
(*                                                                    *)
(* ENTER WHEN: m/tsr = 1, flag0 = 1                                   *)
(*                                                                    *)
(* ACTION :          On Transition             On Match               *)
(* High Pulse     : Start measurement.      Null.                     *) (*V0.3 R.Soja, 6/5/93*)
(* Low Pulse      : Null.                   Null.                     *)
(* Rising Period  : Update measurement.     Update measurement.       *)
(* Falling Period : Null.                   Update measurement.       *)
(*                                                                    *)
(**********************************************************************)
%entry  ram p <- @LAST_TIME_PTA; start_address *; disable_match;
name = pos_trans1;
cond hsr1 = 0, hsr0 = 0, lsr = x, m/tsr = 1, pin = 1, flag0 = 1.

pos_trans1 :
        if @Period_Measure then goto accum_time1_pta, flush. (*V0.3 R.Soja, 6/5/93*)
        if @Match then goto clear_match_pta, flush.

        au chan_reg := chan_reg;  (* It's O.K. to update PSL here since *)
        chan neg_tdl.             (* there are no more Match states.    *)

        if @Low_Pulse then goto End_Of_Phase, flush.

(**************************************************************)
(* Extra code to check for short pulses during flag0=1 states *)
(* in High and Low Pulse measurement functions.               *)
(**************************************************************)
clear_flag_next11_pta :
        if PSL = 0 then goto End_Of_Phase, flush.
        chan clear flag0;
        goto next_match_pta, flush.

clear_flag_next01_pta :
        if PSL = 1 then goto End_Of_Phase, flush.

clear_flag_next_pta :
        chan clear flag0;
        goto next_match_pta, flush.

(*V0.3 Deleted 3 u words, R.Soja, 6/5/93*)

(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  NEG_TRANSITION                                        *)
(*                                                                    *)
(* STATE(S) ENTERED: S3                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : P <- LAST_TIME                                 *)
(*                                                                    *)
(* ENTER WHEN: m/tsr = 1, flag0 = 0                                   *)
(*                                                                    *)
(* ACTION :          On Transition                   On Match         *)
(* High Pulse     : Update measurement.         Invalid.              *)
(* Low Pulse      : Null.                       Update measurement.   *)
(* Rising Period  : Start measurement (Note).   Null.                 *) (*V0.3 R.Soja, 6/5/93*)
(* Falling Period : Start measurement.          Null.                 *)
(*                                                                    *)
(* Note: Caused by short positive pulse.                              *) (*V0.3 R.Soja, 6/5/93*)
(**********************************************************************)
%entry  ram p <- @LAST_TIME_PTA; start_address *; disable_match;
name = neg_trans0;
cond hsr1 = 0, hsr0 = 0, lsr = x, m/tsr = 1, pin = 0, flag0 = 0.

neg_trans0 :

        if @Period_Measure then goto next_match1_pta, flush. (*V0.3 R.Soja, 6/5/93*)
        if @Match then goto accum_time_pta, flush. (* Can only be Low Pulse *)
        au chan_reg := chan_reg;     (* It's O.K. to update PSL here since *)
        chan neg_tdl.                (* there are no more Match states.    *)

(*V0.3 R.Soja, Start ******************************************)
(**************************************************************)
(* Extra code to check for short pulses during flag0=0 states *)
(* in High and Low Pulse measurement functions.               *)
(**************************************************************)
        nop.

low_pulse00_pta :
        if PSL = 1 then goto End_Of_Phase, flush.

low_pulse0_pta :
(*********  Low Pulse, Match, pin=1, flag0=0, is invalid state *********)
(*********  High Pulse, Match, pin=0, flag0=0, is invalid state ********)
(*        if @Match then goto clear_match_pta, flush.                   *)
        chan set flag0;
        goto accum_time_pta, flush.

(*V0.3 R.Soja, End ********************************************)

(**********************************************************************)
(*                                                                    *)
(* ENTRY name:  NEG_TRANSITION                                        *)
(*                                                                    *)
(* STATE(S) ENTERED: S4                                               *)
(*                                                                    *)
(* PRELOAD PARAMETER : P <- LAST_TIME                                 *)
(*                                                                    *)
(* ENTER WHEN: m/tsr = 1, flag0 = 1                                   *)
(*                                                                    *)
(* ACTION :          On Transition             On Match               *)
(* High Pulse     : Null.                   Null.                     *)
(* Low Pulse      : Start measurement.      Null.                     *) (*V0.3 R.Soja, 6/5/93*)
(* Rising Period  : Null.                   Update measurement.       *)
(* Falling Period : Update measurement.     Update measurement.       *)
(*                                                                    *)
(**********************************************************************)
%entry  ram p <- @LAST_TIME_PTA; start_address *; disable_match;
name = neg_trans1;
cond hsr1 = 0, hsr0 = 0, lsr = x, m/tsr = 1, pin = 0, flag0 = 1.

neg_trans1 :
        if @Period_Measure then goto accum_time1_pta, flush. (*V0.3 R.Soja, 6/5/93*)
        if @Match then goto clear_match_pta, flush.

        au chan_reg := chan_reg;  (* It's O.K. to update PSL here since *)
        chan neg_tdl.             (* there are no more Match tests.     *)

        if @High_Pulse then goto End_Of_Phase, flush.
        goto clear_flag_next01_pta, flush.

(*V0.3 Delete 2 u words, R.Soja, 6/5/93*)

clear_trans_pta :

(**** For test purposes only ****)
(*        call disp_state_pta, flush.
        chan cir;*)
(********************************)

        chan neg_tdl;
        end.

clear_match_pta :

(**** For test purposes only ****)
(*        call disp_state_pta, flush.
        chan cir;*)
(********************************)

        chan neg_mrl, neg_lsl;
        end.

(**********************************************************************)
(*    UNUSED ENTRIES - execute appropriate termination                *)
(**********************************************************************)

%entry  start_address End_Of_Link;
name = del_lnk_pta;
cond hsr1 = 0,hsr0 = 0, m/tsr = 0, lsr = 1, flag0 = x, pin = x.

%entry  start_address clear_match_pta;
name = hsr01_pta;
cond hsr1 = 0,hsr0 = 1, pin = x.

%entry  start_address clear_match_pta;
name = hsr10_pta;
cond hsr1 = 1,hsr0 = 0.

(**********************************************************************)
(*                      For test purposes only                        *)
(**********************************************************************)
(*disp_state_pta :
        au chan_reg := chan_reg.
        au p := 0.
        if PSL = 0 then goto pin_done_pta, flush.
        au p_high := 1.

pin_done_pta :
        if flag0 = 0 then goto flag_done_pta, flush.
        au p_low := 1.

flag_done_pta :
        if MRL = 0 then goto write_state_pta, flush.
        au p_high := p_high + #$80.

write_state_pta :
        return.
        ram p -> prm0.*)

(**********************************************************************)
